import {
  Cors,
  LambdaIntegration,
  RestApi,
} from "aws-cdk-lib/aws-apigateway";

import { Stack } from "aws-cdk-lib";
import * as iam from "aws-cdk-lib/aws-iam";
import { Backend } from "./backend";
import { INTERNAL__StatewideServicesKey, INTERNAL__UserMetadataKey } from "./data/resource";
import { env } from "./env.setup";

export const S3_UPLOAD_DIRECTORY_NAME = 'feeds';
export const S3_EXPECTED_LOCATIONS_GEOJSON_FILE_NAME = 'locations.geojson';

const startStepFunctionPolicy = new iam.PolicyStatement({
  sid: "StartStepFunctionPolicy",
  actions: ["states:StartExecution"],
  resources: [env.BACKEND_AWS_STEP_FUNCTION_ARN],
})

const startPreviewToLiveStepFunctionPolicy = new iam.PolicyStatement({
  sid: "StartStepFunctionPolicy",
  actions: ["states:StartExecution"],
  resources: [env.BACKEND_AWS_PUBLISH_CHANGES_STEP_FUNCTION_ARN,],
})


const GROUP_BUCKET_ACCESS_POLICY = new iam.PolicyStatement({
  sid: `${env.BACKEND_AWS_COGNITO_ADMIN_GROUP_NAME}GroupS3BucketAccessPolicy`,
  actions: [
    "s3:PutObject",
    "s3:GetObject",
    "s3:AbortMultipartUpload",
    "s3:ListMultipartUploadParts",
    "kms:Decrypt",
    "kms:GenerateDataKey",
  ],
  resources: [
    `arn:aws:s3:::${env.BACKEND_AWS_S3_BUCKET_NAME}/${S3_UPLOAD_DIRECTORY_NAME}/upload*`,
  ]
})

const GROUP_BUCKET_ACCESS_DOWNLOAD_AND_LIST_POLICY = new iam.PolicyStatement({
  sid: `${env.BACKEND_AWS_COGNITO_ADMIN_GROUP_NAME}GroupS3BucketAccessDownloadAndListPolicy`,
  actions: [
    "s3:GetObject",
  ],
  resources: [
    `arn:aws:s3:::${env.BACKEND_AWS_S3_BUCKET_NAME}/${S3_UPLOAD_DIRECTORY_NAME}/preview*`,
    `arn:aws:s3:::${env.BACKEND_AWS_S3_BUCKET_NAME}/${S3_UPLOAD_DIRECTORY_NAME}/live*`,
  ]
})


const GROUP_BUCKET_GET_POLICY = new iam.PolicyStatement({
  sid: `${env.BACKEND_AWS_COGNITO_ADMIN_GROUP_NAME}GroupS3BucketGetLocationsGeojsonPolicy`,
  actions: ["s3:GetObject"],
  resources: [
    `arn:aws:s3:::${env.BACKEND_AWS_S3_BUCKET_NAME}/${S3_UPLOAD_DIRECTORY_NAME}/${S3_EXPECTED_LOCATIONS_GEOJSON_FILE_NAME}`,
  ]
})

/**
 * Extend backend with custom permissions, auth roles, etc.
 * 
 * @param backend deployed backend service
 */
export function extendBackend(backend: Backend) {
  INTERNAL__extendServicePolicies(backend)
  INTERNAL__extendStatewideLambdaFunction(backend);
  INTERNAL__extendUserLambdaFunction(backend);
  INTERNAL__appendRestApi(backend)
}

function INTERNAL__extendServicePolicies(backend: Backend) {
  backend.auth.resources
    .groups[env.BACKEND_AWS_COGNITO_ADMIN_GROUP_NAME]
    .role
    .addToPrincipalPolicy(GROUP_BUCKET_ACCESS_POLICY)

  backend.auth.resources
    .groups[env.BACKEND_AWS_COGNITO_ADMIN_GROUP_NAME]
    .role
    .addToPrincipalPolicy(GROUP_BUCKET_GET_POLICY)

  backend.auth.resources
    .groups[env.BACKEND_AWS_COGNITO_ADMIN_GROUP_NAME]
    .role
    .addToPrincipalPolicy(GROUP_BUCKET_ACCESS_DOWNLOAD_AND_LIST_POLICY)

  backend.startOtpDataUpdate.resources.lambda.addToRolePolicy(startStepFunctionPolicy)
  backend.triggerPublishPreviewToLive.resources.lambda.addToRolePolicy(startPreviewToLiveStepFunctionPolicy)

}

function INTERNAL__extendStatewideLambdaFunction(backend: Backend) {
  const statewideServicesTable = backend.data.resources.tables[INTERNAL__StatewideServicesKey]
  const statewideServicesStack = Stack.of(statewideServicesTable);

  const dynamodbAccessForLambdaPolicy = new iam.Policy(
    statewideServicesStack,
    `${env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX}DynamoDBAccessForLambdaPolicy`,
    {
      statements: [
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: [
            'dynamodb:Scan',
          ],
          resources: [
            statewideServicesTable.tableArn,
          ]
        })
      ]
    }
  )

  backend.getAllStatewideServices.resources.lambda.role?.attachInlinePolicy(dynamodbAccessForLambdaPolicy)
}

function INTERNAL__extendUserLambdaFunction(backend: Backend) {
  const userMetadataTable = backend.data.resources.tables[INTERNAL__UserMetadataKey]
  const userMetadataStack = Stack.of(userMetadataTable);

  const dynamodbPutAccessForLambdaPolicy = new iam.Policy(
    userMetadataStack,
    `${env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX}DynamoDBPutAccessForLambdaPolicy`,
    {
      statements: [
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: [
            'dynamodb:PutItem',
          ],
          resources: [
            userMetadataTable.tableArn,
          ]
        })
      ]
    }
  )

  const dynamodbDeleteAccessForLambdaPolicy = new iam.Policy(
    userMetadataStack,
    `${env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX}DynamoDBDeleteAccessForLambdaPolicy`,
    {
      statements: [
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: [
            'dynamodb:DeleteItem',
          ],
          resources: [
            userMetadataTable.tableArn,
          ]
        })
      ]
    }
  )


  backend.inviteUser.resources.lambda.role?.attachInlinePolicy(dynamodbPutAccessForLambdaPolicy)
  backend.deleteUser.resources.lambda.role?.attachInlinePolicy(dynamodbDeleteAccessForLambdaPolicy)
}


function INTERNAL__appendRestApi(backend: Backend) {
  const apiStack = backend.createStack(`${env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX}-api-stack`);

  const restApi = new RestApi(apiStack, `${env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX}StatewideServicesRestApiId`, {
    restApiName: `${env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX}StatewideServicesRestApi`,
    deploy: true,
    deployOptions: {
      stageName: env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX,
    },
    defaultCorsPreflightOptions: {
      allowOrigins: Cors.ALL_ORIGINS,
      allowMethods: Cors.ALL_METHODS,
      allowHeaders: Cors.DEFAULT_HEADERS,
    },
  })

  const lambdaIntegration = new LambdaIntegration(
    backend.getAllStatewideServices.resources.lambda
  )

  const servicesPath = restApi.root.addResource("services")
  servicesPath.addMethod("GET", lambdaIntegration);

  servicesPath.addProxy({
    anyMethod: true,
    defaultIntegration: lambdaIntegration
  })

  const apiRestPolicy = new iam.Policy(apiStack, `${env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX}RestApiPolicy`, {
    statements: [
      new iam.PolicyStatement({
        actions: ["execute-api:Invoke"],
        resources: [
          `${restApi.arnForExecuteApi("GET", "/services", env.BACKEND_AWS_COGNITO_RESOURCE_NAME_PREFIX)}`,
        ],
      }),
    ],
  });

  backend.auth.resources.authenticatedUserIamRole.attachInlinePolicy(apiRestPolicy)
  backend.auth.resources.unauthenticatedUserIamRole.attachInlinePolicy(apiRestPolicy)

  backend.addOutput({
    custom: {
      API: {
        [restApi.restApiName]: {
          endpoint: restApi.url,
          region: Stack.of(apiStack).region,
          apiName: restApi.restApiName,
        }
      }
    }
  })
}
